<?php
session_start();
require_once 'includes/auth_check.php';

$page = 'settings';
$database = new Database();
$db = $database->getConnection();

// Initialize variables
$message = '';
$messageType = 'danger';
$user = null;

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

try {
    // Get user information with only the fields we know exist
    $stmt = $db->prepare("
        SELECT id, username, email, first_name, last_name, phone, address, 
               country, gender, date_of_birth, profile_picture, 
               account_type, currency, employment_status, income_source,
               created_at, last_login
        FROM users 
        WHERE id = ?
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        header('Location: ../login.php');
        exit;
    }
    
} catch (Exception $e) {
    error_log("Error loading user settings: " . $e->getMessage());
    $message = "Error loading settings. Please try again.";
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $message = "Invalid form submission. Please try again.";
    } else {
        $action = $_POST['action'] ?? '';
        
        switch ($action) {
            case 'update_profile_picture':
                try {
                    // Handle profile picture upload
                    if (isset($_FILES['profile_picture']) && $_FILES['profile_picture']['error'] === 0) {
                        $uploadDir = '../uploads/profiles/';
                        if (!file_exists($uploadDir)) {
                            mkdir($uploadDir, 0755, true);
                        }
                        
                        // Validate file type
                        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
                        $fileType = $_FILES['profile_picture']['type'];
                        
                        if (!in_array($fileType, $allowedTypes)) {
                            throw new Exception("Only JPG, PNG, and GIF files are allowed.");
                        }
                        
                        // Validate file size (max 2MB)
                        if ($_FILES['profile_picture']['size'] > 2 * 1024 * 1024) {
                            throw new Exception("File size must be less than 2MB.");
                        }
                        
                        $fileName = $_SESSION['user_id'] . '_' . time() . '_' . $_FILES['profile_picture']['name'];
                        $uploadPath = $uploadDir . $fileName;
                        
                        if (move_uploaded_file($_FILES['profile_picture']['tmp_name'], $uploadPath)) {
                            $stmt = $db->prepare("UPDATE users SET profile_picture = ? WHERE id = ?");
                            $stmt->execute([$fileName, $_SESSION['user_id']]);
                            
                            $message = "Profile picture updated successfully!";
                            $messageType = 'success';
                            
                            // Refresh user data
                            $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
                            $stmt->execute([$_SESSION['user_id']]);
                            $user = $stmt->fetch(PDO::FETCH_ASSOC);
                        } else {
                            throw new Exception("Failed to upload profile picture.");
                        }
                    } else {
                        $message = "Please select a profile picture to upload.";
                    }
                } catch (Exception $e) {
                    $message = "Error updating profile picture: " . $e->getMessage();
                }
                break;
                
            case 'change_password':
                try {
                    // Verify current password
                    $stmt = $db->prepare("SELECT password FROM users WHERE id = ?");
                    $stmt->execute([$_SESSION['user_id']]);
                    $currentHash = $stmt->fetchColumn();
                    
                    if (!password_verify($_POST['current_password'], $currentHash)) {
                        $message = "Current password is incorrect.";
                    } elseif ($_POST['new_password'] !== $_POST['confirm_password']) {
                        $message = "New passwords do not match.";
                    } elseif (strlen($_POST['new_password']) < 8) {
                        $message = "New password must be at least 8 characters long.";
                    } else {
                        $newHash = password_hash($_POST['new_password'], PASSWORD_DEFAULT);
                        $stmt = $db->prepare("UPDATE users SET password = ? WHERE id = ?");
                        $stmt->execute([$newHash, $_SESSION['user_id']]);
                        
                        $message = "Password updated successfully!";
                        $messageType = 'success';
                    }
                } catch (Exception $e) {
                    $message = "Error updating password: " . $e->getMessage();
                }
                break;
                
            case 'change_pin':
                try {
                    // Verify current PIN
                    $stmt = $db->prepare("SELECT pin FROM users WHERE id = ?");
                    $stmt->execute([$_SESSION['user_id']]);
                    $currentPin = $stmt->fetchColumn();
                    
                    if ($_POST['current_pin'] !== $currentPin) {
                        $message = "Current PIN is incorrect.";
                    } elseif ($_POST['new_pin'] !== $_POST['confirm_pin']) {
                        $message = "New PINs do not match.";
                    } elseif (!preg_match('/^\d{4}$/', $_POST['new_pin'])) {
                        $message = "PIN must be exactly 4 digits.";
                    } else {
                        $stmt = $db->prepare("UPDATE users SET pin = ? WHERE id = ?");
                        $stmt->execute([$_POST['new_pin'], $_SESSION['user_id']]);
                        
                        $message = "PIN updated successfully!";
                        $messageType = 'success';
                    }
                } catch (Exception $e) {
                    $message = "Error updating PIN: " . $e->getMessage();
                }
                break;
                
            case 'update_contact_info':
                try {
                    // Only allow updating phone and email
                    $stmt = $db->prepare("UPDATE users SET phone = ?, email = ? WHERE id = ?");
                    $stmt->execute([
                        $_POST['phone'],
                        $_POST['email'],
                        $_SESSION['user_id']
                    ]);
                    
                    $message = "Contact information updated successfully!";
                    $messageType = 'success';
                    
                    // Refresh user data
                    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
                    $stmt->execute([$_SESSION['user_id']]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                } catch (Exception $e) {
                    $message = "Error updating contact information: " . $e->getMessage();
                }
                break;
                
            case 'update_notification_preferences':
                try {
                    $stmt = $db->prepare("
                        UPDATE users SET 
                        email_notifications = ?, sms_notifications = ?
                        WHERE id = ?
                    ");
                    
                    $stmt->execute([
                        isset($_POST['email_notifications']) ? 1 : 0,
                        isset($_POST['sms_notifications']) ? 1 : 0,
                        $_SESSION['user_id']
                    ]);
                    
                    $message = "Notification preferences updated successfully!";
                    $messageType = 'success';
                    
                    // Refresh user data
                    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
                    $stmt->execute([$_SESSION['user_id']]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                } catch (Exception $e) {
                    $message = "Error updating notification preferences: " . $e->getMessage();
                }
                break;
        }
    }
}

include 'includes/header.php';
include 'includes/sidebar.php';
?>

<style>
/* Base Styles */
:root {
    --primary-color: #2563eb;
    --secondary-color: #1e40af;
    --success-color: #059669;
    --danger-color: #dc2626;
    --warning-color: #ea580c;
    --info-color: #0284c7;
    --background-color: #f0f2f5;
}

/* Animation Keyframes */
@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

@keyframes slideInUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Animated Elements */
.fade-in {
    animation: fadeIn 0.5s ease forwards;
}

.slide-up {
    animation: slideInUp 0.5s ease forwards;
}

/* Card Styles with Animation */
.dashboard-card {
    background: white;
    border-radius: 15px;
    border: none;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    margin-bottom: 1.5rem;
    transition: all 0.3s ease;
    overflow: hidden;
    position: relative;
}

.dashboard-card:hover {
    box-shadow: 0 15px 30px -10px rgba(0,0,0,0.1), 0 4px 6px -4px rgba(0,0,0,0.1);
    transform: translateY(-5px);
}

.card-header {
    background: none;
    padding: 1.25rem;
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
    z-index: 2;
}

.card-title {
    margin-bottom: 0;
    font-size: 1.1rem;
    font-weight: 600;
    display: flex;
    align-items: center;
}

.card-title i {
    margin-right: 0.5rem;
    font-size: 1.25rem;
    color: var(--primary-color);
}

.card-body {
    padding: 1.5rem;
    position: relative;
    z-index: 1;
}

/* Settings Navigation */
.settings-nav {
    background: white;
    border-radius: 15px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    margin-bottom: 1.5rem;
    overflow: hidden;
}

.nav-tabs {
    border-bottom: none;
    padding: 0.5rem;
}

.nav-tabs .nav-link {
    border: none;
    border-radius: 10px;
    color: #64748b;
    font-weight: 500;
    padding: 0.75rem 1.25rem;
    margin: 0 0.25rem;
    transition: all 0.3s ease;
}

.nav-tabs .nav-link.active {
    background-color: #eef2ff;
    color: var(--primary-color);
}

.nav-tabs .nav-link:hover:not(.active) {
    background-color: #f8fafc;
    color: var(--primary-color);
}

/* Profile Picture Styles */
.profile-picture-container {
    position: relative;
    display: inline-block;
    margin-bottom: 1.5rem;
}

.profile-picture {
    width: 120px;
    height: 120px;
    border-radius: 50%;
    object-fit: cover;
    border: 4px solid #e9ecef;
    transition: all 0.3s ease;
}

.profile-picture:hover {
    border-color: var(--primary-color);
    transform: scale(1.05);
}

.profile-picture-placeholder {
    width: 120px;
    height: 120px;
    border-radius: 50%;
    background-color: #e9ecef;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 3rem;
    color: #6c757d;
    transition: all 0.3s ease;
}

.profile-picture-upload {
    position: absolute;
    bottom: 0;
    right: 0;
    background: var(--primary-color);
    color: white;
    width: 32px;
    height: 32px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 2px 5px rgba(0,0,0,0.2);
}

/* Form Styles */
.form-control, .form-select {
    padding: 0.75rem 1rem;
    border-radius: 10px;
    border: 2px solid #e2e8f0;
    transition: all 0.3s ease;
}

.form-control:focus, .form-select:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 0.25rem rgba(37, 99, 235, 0.1);
}

/* Info Items */
.info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.75rem 0;
    border-bottom: 1px solid #e5e7eb;
}

.info-item:last-child {
    border-bottom: none;
}

/* PIN Input */
.pin-input {
    letter-spacing: 0.5rem;
    font-size: 1.5rem;
    text-align: center;
}

/* Alert Animation */
.alert {
    animation: slideInUp 0.5s ease forwards;
    border-radius: 10px;
}

/* Read-only fields */
.form-control:disabled, .form-control[readonly] {
    background-color: #f8fafc;
    opacity: 1;
}

/* Button styles */
.btn-primary {
    background-color: var(--primary-color);
    border-color: var(--primary-color);
}

.btn-primary:hover {
    background-color: var(--secondary-color);
    border-color: var(--secondary-color);
}

/* Form switch */
.form-switch .form-check-input:checked {
    background-color: var(--primary-color);
    border-color: var(--primary-color);
}
</style>

<div class="main-content">
    <div class="container py-4">
        <div class="d-flex justify-content-between align-items-center mb-4 fade-in">
            <div>
                <h1 class="h3 mb-1">Account Settings</h1>
                <p class="text-muted mb-0">View your account details and manage security settings</p>
            </div>
        </div>
        
        <?php if ($message): ?>
            <div class="alert alert-<?php echo $messageType; ?> alert-dismissible fade show" role="alert">
                <i class='bx bx-<?php echo $messageType === 'success' ? 'check-circle' : 'error-circle'; ?> me-2'></i>
                <?php echo htmlspecialchars($message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        
        <!-- Settings Navigation -->
        <div class="settings-nav slide-up" style="animation-delay: 0.1s;">
            <ul class="nav nav-tabs" id="settingsTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="profile-tab" data-bs-toggle="tab" data-bs-target="#profile" type="button" role="tab" aria-controls="profile" aria-selected="true">
                        <i class='bx bx-user me-2'></i>Profile
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="security-tab" data-bs-toggle="tab" data-bs-target="#security" type="button" role="tab" aria-controls="security" aria-selected="false">
                        <i class='bx bx-shield me-2'></i>Security
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="notifications-tab" data-bs-toggle="tab" data-bs-target="#notifications" type="button" role="tab" aria-controls="notifications" aria-selected="false">
                        <i class='bx bx-bell me-2'></i>Notifications
                    </button>
                </li>
            </ul>
        </div>
        
        <!-- Tab Content -->
        <div class="tab-content" id="settingsTabContent">
            
            <!-- Profile Tab -->
            <div class="tab-pane fade show active" id="profile" role="tabpanel" aria-labelledby="profile-tab">
                <div class="row">
                    <div class="col-lg-8">
                        <div class="dashboard-card slide-up" style="animation-delay: 0.2s;">
                            <div class="card-header">
                                <h5 class="card-title"><i class='bx bx-user-circle'></i> Profile Picture</h5>
                            </div>
                            <div class="card-body">
                                <form method="post" enctype="multipart/form-data" id="profilePictureForm">
                                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                    <input type="hidden" name="action" value="update_profile_picture">
                                    
                                    <!-- Profile Picture -->
                                    <div class="text-center">
                                        <div class="profile-picture-container">
                                            <?php if (!empty($user['profile_picture'])): ?>
                                                <img src="../uploads/profiles/<?php echo htmlspecialchars($user['profile_picture']); ?>" 
                                                     alt="Profile Picture" class="profile-picture" id="profilePreview"
                                                     onerror="this.onerror=null; this.src='../assets/images/default-avatar.png';">
                                            <?php else: ?>
                                                <div class="profile-picture-placeholder" id="profilePreview">
                                                    <?php echo strtoupper(substr($user['first_name'] ?? 'U', 0, 1)); ?>
                                                </div>
                                            <?php endif; ?>
                                            <label for="profile_picture" class="profile-picture-upload">
                                                <i class='bx bx-camera'></i>
                                            </label>
                                            <input type="file" id="profile_picture" name="profile_picture" 
                                                   accept="image/*" style="display: none;">
                                        </div>
                                        <div class="form-text mt-2">JPG, PNG or GIF. Max size 2MB.</div>
                                    </div>
                                    
                                    <div class="text-center mt-3">
                                        <button type="submit" class="btn btn-primary">
                                            <i class='bx bx-save me-2'></i> Update Profile Picture
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                        
                        <div class="dashboard-card slide-up" style="animation-delay: 0.3s;">
                            <div class="card-header">
                                <h5 class="card-title"><i class='bx bx-envelope'></i> Contact Information</h5>
                            </div>
                            <div class="card-body">
                                <form method="post" id="contactForm">
                                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                    <input type="hidden" name="action" value="update_contact_info">
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Email Address</label>
                                        <input type="email" class="form-control" name="email" 
                                               value="<?php echo htmlspecialchars($user['email'] ?? ''); ?>" required>
                                        <div class="form-text">This email will be used for account notifications and communications.</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Phone Number</label>
                                        <input type="tel" class="form-control" name="phone" 
                                               value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>"
                                               placeholder="(555) 123-4567">
                                        <div class="form-text">This phone number will be used for account security and notifications.</div>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-primary">
                                        <i class='bx bx-save me-2'></i> Update Contact Information
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-lg-4">
                        <!-- Account Info -->
                        <div class="dashboard-card slide-up" style="animation-delay: 0.4s;">
                            <div class="card-header">
                                <h5 class="card-title"><i class='bx bx-info-circle'></i> Account Information</h5>
                            </div>
                            <div class="card-body">
                                <div class="info-item">
                                    <span>Full Name:</span>
                                    <span class="fw-bold"><?php echo htmlspecialchars(($user['first_name'] ?? '') . ' ' . ($user['last_name'] ?? '')); ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Username:</span>
                                    <span class="fw-bold"><?php echo htmlspecialchars($user['username'] ?? 'N/A'); ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Account Type:</span>
                                    <span class="fw-bold"><?php echo htmlspecialchars(ucfirst($user['account_type'] ?? 'Standard')); ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Currency:</span>
                                    <span class="fw-bold"><?php echo htmlspecialchars($user['currency'] ?? 'USD'); ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Member Since:</span>
                                    <span><?php echo isset($user['created_at']) ? date('M j, Y', strtotime($user['created_at'])) : 'N/A'; ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Last Login:</span>
                                    <span><?php echo isset($user['last_login']) && $user['last_login'] ? date('M j, Y g:i A', strtotime($user['last_login'])) : 'Never'; ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Account Status:</span>
                                    <span class="badge bg-success">Active</span>
                                </div>
                                
                                <div class="alert alert-info mt-3" role="alert">
                                    <i class='bx bx-info-circle me-2'></i>
                                    To update your personal information, please contact customer support or visit your nearest branch.
                                </div>
                            </div>
                        </div>
                        
                        <!-- Personal Info -->
                        <div class="dashboard-card slide-up" style="animation-delay: 0.5s;">
                            <div class="card-header">
                                <h5 class="card-title"><i class='bx bx-user'></i> Personal Details</h5>
                            </div>
                            <div class="card-body">
                                <div class="info-item">
                                    <span>Date of Birth:</span>
                                    <span><?php echo isset($user['date_of_birth']) && $user['date_of_birth'] ? date('M j, Y', strtotime($user['date_of_birth'])) : 'Not provided'; ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Gender:</span>
                                    <span><?php echo htmlspecialchars(ucfirst($user['gender'] ?? 'Not provided')); ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Address:</span>
                                    <span><?php echo htmlspecialchars($user['address'] ?? 'Not provided'); ?></span>
                                </div>
                                <div class="info-item">
                                    <span>Country:</span>
                                    <span><?php echo htmlspecialchars($user['country'] ?? 'Not provided'); ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Security Tab -->
            <div class="tab-pane fade" id="security" role="tabpanel" aria-labelledby="security-tab">
                <div class="row">
                    <div class="col-lg-6">
                        <!-- Change Password -->
                        <div class="dashboard-card slide-up" style="animation-delay: 0.2s;">
                            <div class="card-header">
                                <h5 class="card-title"><i class='bx bx-key'></i> Change Password</h5>
                            </div>
                            <div class="card-body">
                                <form method="post" id="passwordForm">
                                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                    <input type="hidden" name="action" value="change_password">
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Current Password</label>
                                        <div class="input-group">
                                            <input type="password" class="form-control" name="current_password" 
                                                   id="currentPassword" required>
                                            <button class="btn btn-outline-secondary" type="button" 
                                                    onclick="togglePassword('currentPassword')">
                                                <i class='bx bx-show'></i>
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">New Password</label>
                                        <div class="input-group">
                                            <input type="password" class="form-control" name="new_password" 
                                                   id="newPassword" minlength="8" required>
                                            <button class="btn btn-outline-secondary" type="button" 
                                                    onclick="togglePassword('newPassword')">
                                                <i class='bx bx-show'></i>
                                            </button>
                                        </div>
                                        <div class="form-text">
                                            Minimum 8 characters, at least one letter and one number
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Confirm New Password</label>
                                        <input type="password" class="form-control" name="confirm_password" 
                                               id="confirmPassword" required>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-primary">
                                        <i class='bx bx-key me-2'></i> Update Password
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-lg-6">
                        <!-- Change PIN -->
                        <div class="dashboard-card slide-up" style="animation-delay: 0.3s;">
                            <div class="card-header">
                                <h5 class="card-title"><i class='bx bx-lock'></i> Change PIN</h5>
                            </div>
                            <div class="card-body">
                                <form method="post" id="pinForm">
                                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                    <input type="hidden" name="action" value="change_pin">
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Current PIN</label>
                                        <input type="password" class="form-control pin-input" name="current_pin" 
                                               maxlength="4" pattern="[0-9]{4}" inputmode="numeric" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">New PIN</label>
                                        <input type="password" class="form-control pin-input" name="new_pin" 
                                               maxlength="4" pattern="[0-9]{4}" inputmode="numeric" required>
                                        <div class="form-text">PIN must be exactly 4 digits</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Confirm New PIN</label>
                                        <input type="password" class="form-control pin-input" name="confirm_pin" 
                                               maxlength="4" pattern="[0-9]{4}" inputmode="numeric" required>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-primary">
                                        <i class='bx bx-lock me-2'></i> Update PIN
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Security Tips -->
                <div class="dashboard-card slide-up" style="animation-delay: 0.4s;">
                    <div class="card-header">
                        <h5 class="card-title"><i class='bx bx-shield-quarter'></i> Security Tips</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <div class="d-flex">
                                                                      <div class="me-3 text-primary">
                                        <i class='bx bx-lock-alt' style="font-size: 24px;"></i>
                                    </div>
                                    <div>
                                        <h6>Use Strong Passwords</h6>
                                        <p class="text-muted small">Create passwords with a mix of letters, numbers, and special characters.</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6 mb-3">
                                <div class="d-flex">
                                    <div class="me-3 text-warning">
                                        <i class='bx bx-wifi' style="font-size: 24px;"></i>
                                    </div>
                                    <div>
                                        <h6>Secure Networks</h6>
                                        <p class="text-muted small">Only access your account on secure, private networks.</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6 mb-3">
                                <div class="d-flex">
                                    <div class="me-3 text-danger">
                                        <i class='bx bx-link-alt' style="font-size: 24px;"></i>
                                    </div>
                                    <div>
                                        <h6>Beware of Phishing</h6>
                                        <p class="text-muted small">Never click suspicious links or provide your credentials via email.</p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6 mb-3">
                                <div class="d-flex">
                                    <div class="me-3 text-success">
                                        <i class='bx bx-time' style="font-size: 24px;"></i>
                                    </div>
                                    <div>
                                        <h6>Regular Updates</h6>
                                        <p class="text-muted small">Change your password and PIN regularly for enhanced security.</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="alert alert-warning mt-3" role="alert">
                            <i class='bx bx-info-circle me-2'></i>
                            We will never call you to ask for your password, PIN, or OTP. If you receive such calls, please report them immediately.
                        </div>
                    </div>
                </div>
                
             <!-- Login History -->
<div class="table-responsive">
    <table class="table table-hover">
        <thead>
            <tr>
                <th>Date & Time</th>
                <th>IP Address</th>
                <th>Device</th>
                <th>Status</th>
            </tr>
        </thead>
        <tbody>
            <?php
            // Fetch login history if available
            try {
                // Check if the login_history table exists
                $table_check = $db->prepare("
                    SELECT COUNT(*) 
                    FROM information_schema.tables 
                    WHERE table_schema = DATABASE() 
                    AND table_name = 'login_history'
                ");
                $table_check->execute();
                $table_exists = $table_check->fetchColumn() > 0;
                
                if ($table_exists) {
                    // Count records for this user
                    $count_stmt = $db->prepare("
                        SELECT COUNT(*) 
                        FROM login_history 
                        WHERE user_id = ?
                    ");
                    $count_stmt->execute([$_SESSION['user_id']]);
                    $record_count = $count_stmt->fetchColumn();
                    
                    if ($record_count > 0) {
                        $login_stmt = $db->prepare("
                            SELECT login_time, ip_address, device_info, status
                            FROM login_history
                            WHERE user_id = ?
                            ORDER BY login_time DESC
                            LIMIT 5
                        ");
                        $login_stmt->execute([$_SESSION['user_id']]);
                        $login_history = $login_stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        foreach ($login_history as $login):
                        ?>
                        <tr>
                            <td><?php echo date('M j, Y g:i A', strtotime($login['login_time'])); ?></td>
                            <td><?php echo htmlspecialchars($login['ip_address']); ?></td>
                            <td><?php echo htmlspecialchars($login['device_info']); ?></td>
                            <td>
                                <?php if ($login['status'] === 'success'): ?>
                                    <span class="badge bg-success">Success</span>
                                <?php else: ?>
                                    <span class="badge bg-danger">Failed</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php
                        endforeach;
                    } else {
                        // No records for this user
                        ?>
                        <tr>
                            <td colspan="4" class="text-center py-3">
                                <i class='bx bx-info-circle me-2'></i> No login history available for your account yet.
                                <br><small class="text-muted">Login history will be recorded on your next login.</small>
                            </td>
                        </tr>
                        <?php
                        // Show current session as a fallback
                        $current_time = date('M j, Y g:i A');
                        $ip = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
                        $device = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown Device';
                        ?>
                        <tr>
                            <td><?php echo $current_time; ?></td>
                            <td><?php echo htmlspecialchars($ip); ?></td>
                            <td><?php echo htmlspecialchars(substr($device, 0, 50) . '...'); ?></td>
                            <td><span class="badge bg-primary">Current Session</span></td>
                        </tr>
                        <?php
                    }
                } else {
                    // Table doesn't exist
                    ?>
                    <tr>
                        <td colspan="4" class="text-center py-3">
                            <i class='bx bx-error-circle me-2'></i> Login history tracking is not properly configured.
                            <br><small class="text-muted">Please contact the administrator.</small>
                        </td>
                    </tr>
                    <?php
                    // Show current session as a fallback
                    $current_time = date('M j, Y g:i A');
                    $ip = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
                    $device = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown Device';
                    ?>
                    <tr>
                        <td><?php echo $current_time; ?></td>
                        <td><?php echo htmlspecialchars($ip); ?></td>
                        <td><?php echo htmlspecialchars(substr($device, 0, 50) . '...'); ?></td>
                        <td><span class="badge bg-primary">Current Session</span></td>
                    </tr>
                    <?php
                }
            } catch (Exception $e) {
                ?>
                <tr>
                    <td colspan="4" class="text-center py-3">
                        <i class='bx bx-error-circle me-2'></i> Error loading login history
                        <br><small class="text-muted">Error: <?php echo htmlspecialchars($e->getMessage()); ?></small>
                    </td>
                </tr>
                <?php
                // Show current session as a fallback
                $current_time = date('M j, Y g:i A');
                $ip = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
                $device = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown Device';
                ?>
                <tr>
                    <td><?php echo $current_time; ?></td>
                    <td><?php echo htmlspecialchars($ip); ?></td>
                    <td><?php echo htmlspecialchars(substr($device, 0, 50) . '...'); ?></td>
                    <td><span class="badge bg-primary">Current Session</span></td>
                </tr>
                <?php
            }
            ?>
        </tbody>
    </table>
</div>


            </div>
            
            <!-- Notifications Tab -->
            <div class="tab-pane fade" id="notifications" role="tabpanel" aria-labelledby="notifications-tab">
                <div class="dashboard-card slide-up" style="animation-delay: 0.2s;">
                    <div class="card-header">
                        <h5 class="card-title"><i class='bx bx-bell'></i> Notification Preferences</h5>
                    </div>
                    <div class="card-body">
                        <form method="post" id="notificationsForm">
                            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                            <input type="hidden" name="action" value="update_notification_preferences">
                            
                            <div class="mb-4">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <div>
                                        <h6 class="mb-1">
                                            <i class='bx bx-envelope me-2'></i>Email Notifications
                                        </h6>
                                        <p class="text-muted mb-0">Receive account updates and alerts via email</p>
                                    </div>
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" name="email_notifications" 
                                               id="emailNotifications" <?php echo isset($user['email_notifications']) && $user['email_notifications'] ? 'checked' : ''; ?>>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-4">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <div>
                                        <h6 class="mb-1">
                                            <i class='bx bx-message me-2'></i>SMS Notifications
                                        </h6>
                                        <p class="text-muted mb-0">Receive important alerts via text message</p>
                                    </div>
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" name="sms_notifications" 
                                               id="smsNotifications" <?php echo isset($user['sms_notifications']) && $user['sms_notifications'] ? 'checked' : ''; ?>>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-4">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <div>
                                        <h6 class="mb-1">
                                            <i class='bx bx-credit-card me-2'></i>Transaction Alerts
                                        </h6>
                                        <p class="text-muted mb-0">Get notified of all account transactions</p>
                                    </div>
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" name="transaction_alerts" 
                                               id="transactionAlerts" checked>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-4">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <div>
                                        <h6 class="mb-1">
                                            <i class='bx bx-shield me-2'></i>Security Alerts
                                        </h6>
                                        <p class="text-muted mb-0">Receive notifications about security events</p>
                                    </div>
                                    <div class="form-check form-switch">
                                        <input class="form-check-input" type="checkbox" name="security_alerts" 
                                               id="securityAlerts" checked disabled>
                                    </div>
                                </div>
                                <small class="text-muted">Security alerts cannot be disabled for your protection</small>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">
                                <i class='bx bx-save me-2'></i> Save Preferences
                            </button>
                        </form>
                    </div>
                </div>
                
                <!-- Communication Preferences -->
                <div class="dashboard-card slide-up" style="animation-delay: 0.3s;">
                    <div class="card-header">
                        <h5 class="card-title"><i class='bx bx-envelope'></i> Communication Preferences</h5>
                    </div>
                    <div class="card-body">
                        <div class="mb-4">
                            <h6>Statement Delivery</h6>
                            <div class="form-check mb-2">
                                <input class="form-check-input" type="radio" name="statementDelivery" id="emailStatement" checked>
                                <label class="form-check-label" for="emailStatement">
                                    Email (Free)
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="statementDelivery" id="paperStatement">
                                <label class="form-check-label" for="paperStatement">
                                    Paper Mail (Fee may apply)
                                </label>
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <h6>Marketing Communications</h6>
                            <div class="form-check mb-2">
                                <input class="form-check-input" type="checkbox" id="promotionalEmails" checked>
                                <label class="form-check-label" for="promotionalEmails">
                                    Promotional Emails
                                </label>
                            </div>
                            <div class="form-check mb-2">
                                <input class="form-check-input" type="checkbox" id="productUpdates" checked>
                                <label class="form-check-label" for="productUpdates">
                                    Product Updates
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="partnerOffers">
                                <label class="form-check-label" for="partnerOffers">
                                    Partner Offers
                                </label>
                            </div>
                        </div>
                        
                        <button type="button" class="btn btn-primary">
                            <i class='bx bx-save me-2'></i> Save Preferences
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize Bootstrap tabs
    var triggerTabList = [].slice.call(document.querySelectorAll('#settingsTabs button'));
    triggerTabList.forEach(function (triggerEl) {
        var tabTrigger = new bootstrap.Tab(triggerEl);
        
        triggerEl.addEventListener('click', function (event) {
            event.preventDefault();
            tabTrigger.show();
        });
    });
    
    // Profile picture preview
    document.getElementById('profile_picture').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            // Validate file size (max 2MB)
            if (file.size > 2 * 1024 * 1024) {
                alert('File size must be less than 2MB.');
                this.value = '';
                return;
            }
            
            // Validate file type
            const validTypes = ['image/jpeg', 'image/png', 'image/gif'];
            if (!validTypes.includes(file.type)) {
                alert('Only JPG, PNG, and GIF files are allowed.');
                this.value = '';
                return;
            }
            
            const reader = new FileReader();
            reader.onload = function(e) {
                const preview = document.getElementById('profilePreview');
                if (preview.tagName === 'IMG') {
                    preview.src = e.target.result;
                } else {
                    const img = document.createElement('img');
                    img.src = e.target.result;
                    img.alt = "Profile Picture";
                    img.className = "profile-picture";
                    img.id = "profilePreview";
                    preview.parentNode.replaceChild(img, preview);
                }
            };
            reader.readAsDataURL(file);
            
            // Auto-submit the form when a file is selected
            document.getElementById('profilePictureForm').submit();
        }
    });
    
    // Toggle password visibility
    window.togglePassword = function(fieldId) {
        const field = document.getElementById(fieldId);
        const icon = field.nextElementSibling.querySelector('i');
        
        if (field.type === 'password') {
            field.type = 'text';
            icon.className = 'bx bx-hide';
        } else {
            field.type = 'password';
            icon.className = 'bx bx-show';
        }
    };
    
    // PIN input formatting
    const pinInputs = document.querySelectorAll('.pin-input');
    pinInputs.forEach(input => {
        input.addEventListener('input', function() {
            this.value = this.value.replace(/[^0-9]/g, '').substring(0, 4);
        });
    });
    
    // End all sessions function
    window.endAllSessions = function() {
        if (confirm('This will log you out from all devices. Continue?')) {
            // Create a form and submit it
            const form = document.createElement('form');
            form.method = 'post';
            form.innerHTML = `
                <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                <input type="hidden" name="action" value="end_all_sessions">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    };
    
    // Form validation
    const passwordForm = document.getElementById('passwordForm');
    if (passwordForm) {
        passwordForm.addEventListener('submit', function(e) {
            const newPassword = document.getElementById('newPassword').value;
            const confirmPassword = document.getElementById('confirmPassword').value;
            
            if (newPassword !== confirmPassword) {
                e.preventDefault();
                alert('New passwords do not match!');
            }
            
            // Check password strength
            if (!/^(?=.*[A-Za-z])(?=.*\d)[A-Za-z\d]{8,}$/.test(newPassword)) {
                e.preventDefault();
                alert('Password must be at least 8 characters with at least one letter and one number.');
            }
        });
    }
    
    const pinForm = document.getElementById('pinForm');
    if (pinForm) {
        pinForm.addEventListener('submit', function(e) {
            const newPin = document.querySelector('input[name="new_pin"]').value;
            const confirmPin = document.querySelector('input[name="confirm_pin"]').value;
            
            if (newPin !== confirmPin) {
                e.preventDefault();
                alert('New PINs do not match!');
            }
            
            if (!/^\d{4}$/.test(newPin)) {
                e.preventDefault();
                alert('PIN must be exactly 4 digits!');
            }
        });
    }
    
    // Phone number formatting
    const phoneInput = document.querySelector('input[name="phone"]');
    if (phoneInput) {
        phoneInput.addEventListener('input', function() {
            let cleaned = this.value.replace(/\D/g, '');
            let formatted = cleaned;
            
            if (cleaned.length > 0) {
                if (cleaned.length <= 3) {
                    formatted = cleaned;
                } else if (cleaned.length <= 6) {
                    formatted = `(${cleaned.slice(0,3)}) ${cleaned.slice(3)}`;
                } else {
                    formatted = `(${cleaned.slice(0,3)}) ${cleaned.slice(3,6)}-${cleaned.slice(6,10)}`;
                }
            }
            
            this.value = formatted;
        });
    }
    
    // Check if there's a hash in the URL and activate the corresponding tab
    const hash = window.location.hash;
    if (hash) {
        const tabId = hash.replace('#', '');
        const tab = document.querySelector(`#settingsTabs button[data-bs-target="#${tabId}"]`);
        if (tab) {
            const bsTab = new bootstrap.Tab(tab);
            bsTab.show();
        }
    }
});
</script>

<?php include 'includes/footer.php'; ?>

