<?php
/**
 * Get unread notifications count for a user
 * 
 * @param PDO $db Database connection
 * @param int $user_id User ID
 * @return int Number of unread notifications
 */
function getUnreadNotificationsCount($db, $user_id) {
    try {
        $stmt = $db->prepare("SELECT COUNT(*) FROM notifications WHERE user_id = ? AND is_read = 0");
        $stmt->execute([$user_id]);
        return $stmt->fetchColumn();
    } catch (PDOException $e) {
        error_log("Error getting unread notifications count: " . $e->getMessage());
        return 0;
    }
}

/**
 * Get recent notifications for a user
 * 
 * @param PDO $db Database connection
 * @param int $user_id User ID
 * @param int $limit Maximum number of notifications to return
 * @return array Array of notification objects
 */
function getRecentNotifications($db, $user_id, $limit = 5) {
    try {
        $stmt = $db->prepare("
            SELECT * FROM notifications 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$user_id, $limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting recent notifications: " . $e->getMessage());
        return [];
    }
}

/**
 * Mark all notifications as read for a user
 * 
 * @param PDO $db Database connection
 * @param int $user_id User ID
 * @return bool True if successful, false otherwise
 */
function markAllNotificationsAsRead($db, $user_id) {
    try {
        $stmt = $db->prepare("UPDATE notifications SET is_read = 1 WHERE user_id = ?");
        $stmt->execute([$user_id]);
        return true;
    } catch (PDOException $e) {
        error_log("Error marking notifications as read: " . $e->getMessage());
        return false;
    }
}

/**
 * Mark a specific notification as read
 * 
 * @param PDO $db Database connection
 * @param int $notification_id Notification ID
 * @param int $user_id User ID (for security)
 * @return bool True if successful, false otherwise
 */
function markNotificationAsRead($db, $notification_id, $user_id) {
    try {
        $stmt = $db->prepare("UPDATE notifications SET is_read = 1 WHERE id = ? AND user_id = ?");
        $stmt->execute([$notification_id, $user_id]);
        return true;
    } catch (PDOException $e) {
        error_log("Error marking notification as read: " . $e->getMessage());
        return false;
    }
}

/**
 * Get notification icon based on type
 * 
 * @param string $type Notification type
 * @return string Icon class
 */
function getNotificationIcon($type) {
    switch ($type) {
        case 'ticket_reply':
            return 'bx-message-detail';
        case 'ticket_status':
            return 'bx-check-circle';
        case 'ticket_created':
            return 'bx-plus-circle';
        case 'transaction':
            return 'bx-transfer';
        case 'account':
            return 'bx-user';
        default:
            return 'bx-bell';
    }
}

/**
 * Get notification color based on type
 * 
 * @param string $type Notification type
 * @return string Color name
 */
function getNotificationColor($type) {
    switch ($type) {
        case 'ticket_reply':
            return 'primary';
        case 'ticket_status':
            return 'success';
        case 'ticket_created':
            return 'info';
        case 'transaction':
            return 'warning';
        case 'account':
            return 'danger';
        default:
            return 'primary';
    }
}
?>
